;; Routines for "what you see is DocBook XML".
;; copyright 2003 Junichi Uekawa.

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; readme-debian.el is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with your Debian installation, in /usr/share/common-licenses/GPL
;; If not, write to the Free Software Foundation, 675 Mass Ave,
;; Cambridge, MA 02139, USA.

(require 'comint)
(require 'elserv)
(require 'mcharset)

(defvar wysidocbookxml-daemon-name "@LIBEXECDIR@/wysidocbookdaemon" 
  "The default value for wysidocbookxml daemon")
(defconst wysidocbookxml-daemon-process-name "wysidocbookxmldaemon-interactive")
(defconst wysidocbookxml-version "@VERSION@"
  "The version number of wysidocbookxml")
(defvar wysidocbookxml-elserv-port 8090 "Elserv port used for wysidocbookxml")
(defvar wysidocbookxml-elserv-process nil "The elserv process identifier for the elserv process being used for wysidocbookxml.")
(defvar wysidocbookxml-current-buffer nil 
  "The buffer which is going to be displayed through elserv,
for preview.")
(defvar wysidocbookxml-interactive-time 1
  "The number of seconds to wait before renew, slow machines may want
several seconds, but fast machines can have less, to make speed")
(defvar wysidocbookxml-xsl-stylesheet "/usr/share/sgml/docbook/stylesheet/xsl/ldp/ldp-html.xsl"
  "filename of stylesheet to use")
(defvar wysidocbookxml-daemon-buffer nil
  "The buffer that the daemon of wysidocbookdaemon works in ")

(defvar wysidocbookxml-active-modes-alist 
  '((xml-mode . t))
  "*List of major modes that wysidocbookxml mode is activated")
(defvar wysidocbookxml-slice-text
  "chapter"
  "The grade of slicing for wysidocbookxml, chapter is probably good enough.")

(defun wysidocbookxml-start-daemon ()
  "Start up daemon for wysidocbookxml, and assign a buffer for the running process"
  (setq wysidocbookxml-daemon-buffer 
	(make-comint wysidocbookxml-daemon-process-name
		     wysidocbookxml-daemon-name
		     nil "-x" wysidocbookxml-xsl-stylesheet "-s" (number-to-string wysidocbookxml-interactive-time))))

(defun wysidocbookxml-restart-daemon ()
  "Restart the daemon after reconfiguration"
  (interactive)
  (kill-buffer wysidocbookxml-daemon-buffer)
  (wysidocbookxml-start-daemon))

(defun wysidocbookxml-start-elserv ()
  "Internal routine to start up elserv process.
wysidocbookxml uses this process for providing XML portions to xsltproc."
  (setq wysidocbookxml-elserv-process (elserv-start wysidocbookxml-elserv-port))
  (elserv-publish wysidocbookxml-elserv-process
		  "/preview"
		  :allow '("localhost")
		  :function 'wysidocbookxml-elserv-preview
		  :description "Preview of current XML buffer"))

(defun wysidocbookxml-insert-xml-start ()
  "parses STRING and inserts the required tags"
  (insert "<book>"))
  
(defun wysidocbookxml-insert-xml-end ()
  "parses STRING and inserts the required tags"
  (insert "</book>"))

(defun wysidocbookxml-elserv-preview (result path ppath request) 
  "View XML slice from buffer.

RESULT is the resulting value
PATH is relative path from the published path
PPATH is the published path
REQUEST is the request data."
  (save-excursion
    (let* ((charset nil)
	   start-slice end-slice text-string header-end header-string)
      (set-buffer wysidocbookxml-current-buffer)
      (save-excursion 
	
	(setq start-slice (re-search-backward (concat  "<" wysidocbookxml-slice-text "[^>]*>") nil t nil))
	(setq end-slice (re-search-forward (concat  "</" wysidocbookxml-slice-text "[^>]*>") nil t nil))
	(goto-line 1)
	(setq header-end (- (re-search-forward "<book>" nil t nil) 6))
	(setq text-string (buffer-substring-no-properties start-slice end-slice))
	(setq header-string (buffer-substring-no-properties 1 header-end))
	      
	(with-temp-buffer
	  (insert header-string)
	  (wysidocbookxml-insert-xml-start)
	  (insert text-string)
	  (wysidocbookxml-insert-xml-end)
	  (setq charset (detect-mime-charset-region (point-min)(point-max)))
	  (elserv-set-result-header 
	   result
	   (list 'content-type (concat "text/xml; charset=" (symbol-name charset))))
	  (elserv-set-result-body result 
				  (encode-mime-charset-string (buffer-string) charset)))))))

(define-minor-mode wysidocbookxml-mode
  "WYSI Docbook XML minor-mode for auto-previewing of DocBook/XML
through mozilla remote interface"
  nil
  " WYSIDBXML"
  nil
  (if wysidocbookxml-mode
      (progn
	(if (assoc major-mode wysidocbookxml-active-modes-alist)
	    t
	  (setq wysidocbookxml-mode nil)
	  (error "current major-mode is not in wysidocbookxml-active-modes-alist"))
	(if (eq nil wysidocbookxml-daemon-buffer)
	    (wysidocbookxml-start-daemon))
	(wysidocbookxml-daemon-post-command)	;run this once immediately
	(add-hook 'post-command-hook (function wysidocbookxml-daemon-post-command)
		  nil t))
    (progn
      (remove-hook 'post-command-hook (function wysidocbookxml-daemon-post-command)))))

(defun wysidocbookxml-daemon-post-command ()
  "wysidocbookxml that is invoked after post-command-hook"
  (if (not wysidocbookxml-mode)
      t
    (if (local-variable-p 'wysidocbookxml-daemon-buffer-modified-tick)
	t
      (make-local-variable 'wysidocbookxml-daemon-buffer-modified-tick))
    (if (assoc major-mode wysidocbookxml-active-modes-alist)
	(let* ((curval (buffer-modified-tick)))
	  ;; check if all conditions are met.
	  (if (and (boundp 'wysidocbookxml-daemon-buffer-modified-tick)
		   (= curval wysidocbookxml-daemon-buffer-modified-tick)
		   (not (assoc last-command '((recenter . t)))))
	      t				;nothing should be done.
	    ;; otherwise, queue for the redrawing of the page.
	    (setq wysidocbookxml-daemon-buffer-modified-tick curval) 
	    (wysidocbookxml-daemon-preview-page))))))

(defun wysidocbookxml-daemon-preview-page ()
  "Preview the current buffer through xsltproc and mozilla, 
by slicing the current file and providing it through"
  (if (or (eq nil wysidocbookxml-elserv-process)
	  (eq 'exit (process-status wysidocbookxml-elserv-process)))
      (wysidocbookxml-start-elserv))
  (setq wysidocbookxml-current-buffer (current-buffer))
  (save-excursion
    (let* ((deactivate-mark nil)
	   (optional-name "#"))
      (if (re-search-backward " [iI][dD]=\"?\\([^>\" ]*\\)" nil t nil)
	  (setq optional-name (concat "#" (match-string 1))))
      (set-buffer wysidocbookxml-daemon-buffer)
      (insert 
       (concat "http://localhost:" (number-to-string wysidocbookxml-elserv-port) "/preview/index.xml " 
	       optional-name))
      (comint-send-input))))

(provide 'wysidocbookxml)

