/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * structure tree viewer: gives you a clue on tree structure of dmachinemon
 */
/* 
 * 2001 Sep 12
 * $Id: treeview.c,v 1.11 2002/05/22 11:11:17 dancer Exp $
 *
 */


#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <locale.h>
#include <pthread.h>

#include "config.h"
#include "dmachinemon/dmachinemon.h"

#define PROGRAMNAME "treeview"

char * hostname ;
char * portnumber;

static void * 
malloc_with_error (int size)
{
  void * m = malloc (size);
  if (!m )
    {
      fprintf (stderr, PROGRAMNAME ": out of memory\n");
      exit (EXIT_FAILURE);
    }
  return m;
}

static void
printspace(int c)
{
  while ( c-- )
    printf(" ");
}

typedef struct machinetree
{
  char * myname;
  struct machinetree * nextlink;
  struct machinetree * downlink;
} machinetree;

/* getcar */
static char *
getcar (const char * ch)
{
  char * b = ch?strrchr(ch, ','):NULL;
  if (b) b++;
  return b?strdup(b):(ch?strdup(ch):NULL);
}

/* getcdr */
static char*
getcdr (const char * ch)
{
  char * s = ch?strdup(ch):NULL;
  char * b = s?strrchr (s, ','):NULL;
  if (b) 
    *b = 0;
  else 
    {
      free (s);
      s=NULL;
    }
  return s;
}

static void 
processinternal (machinetree * t, char * data)
{
  machinetree * mydata ;
  int found;
  char * buf = getcar (data);
  if (!data)
    return;

  found = 0;
  mydata = t -> downlink;
  while (mydata)
    {
      if (!strcmp(mydata -> myname, buf))
	{
	  found = 1 ;      
	  break;
	}
      mydata = mydata -> nextlink;
    }
  if (!found)
    {
      mydata = malloc(sizeof (machinetree));
      mydata -> downlink = NULL;
      mydata -> myname = buf;
      mydata -> nextlink = t -> downlink;
      t->downlink = mydata;
    }
  else
    free (buf);
  processinternal(mydata, getcdr(data));
  free (data);
}

static machinetree * 
build_local_route_tree (void)
{
  dm_machinelist * new = NULL;
  machinetree * treeptr = NULL;

  /* creating the tree */
  new = maininfo.machinedb;
  treeptr = malloc_with_error (sizeof(machinetree));
  treeptr -> myname = NULL;
  treeptr -> nextlink = treeptr -> downlink = NULL;

  while(new)
    {
      const char * buf;
      buf = dm_get_value_text(new, "Seen-By");
      if (buf)
	{
	  if (!treeptr->myname)
	    treeptr->myname = getcar (buf);
	  if (strchr (buf, ','))
	    processinternal (treeptr,getcdr(buf));
	}
      new = new -> next;
    }  
  return treeptr;  
}


static void access_master(void)
{
  int fd = dm_connect_to_host(hostname, atoi(portnumber));
  FILE* f=(fd!=-1)?fdopen (fd, "r"):NULL;
  
  if (!f)
    {
      fprintf (stderr, PROGRAMNAME ": cannot open server socket\n");
      exit(1);
    }

  if (dm_process_nodes_input (f, &maininfo, NULL, NULL))
    {
      fprintf (stderr, PROGRAMNAME ": could not handle input...\n");
      exit (1);
    }  

  fclose(f);
}


static void
display_local_route_tree(machinetree* m, int level)
{
  while (m)
    {
      printspace (level * 2);
      printf ("%s\n", m->myname);
      
      if (m->downlink)
	display_local_route_tree(m->downlink, level + 1);
      
      m=m->nextlink;
    }
  
}

static void
processing_command (void)
{
  machinetree * m;
  
  access_master();
  m = build_local_route_tree();
  display_local_route_tree(m,0);
  
  return ;  
}

int main(int ac, char ** av)
{ 
  dm_commandoption cdat;
  setlocale(LC_TIME, "");

  dmachinemon_parse_options(ac, av, &cdat);
  if (!cdat.parenthostname)
    {
      fprintf (stderr, 
	       "treeview v. %s \n\n"
	       "Not enough command-line information specified\n\n"
	       ,VERSION
	       );
      dmachinemon_print_commandline_help();
      return 1;
    }

  hostname=cdat.parenthostname;
  portnumber=cdat.port_client;

  processing_command();  
  return 0;
}
