/*
 *  machine monitor on DNAS / a distributed machine monitor by dancer.
 *  Copyright (C) 2001,2002 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * The machine node monitoring system, monitors most load.
 *
 * $Id: machinemon.c,v 1.1 2002/06/14 11:36:51 dancer Exp $
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <pthread.h>
#include "dmachinemon/dmachinemon-debug.h"
#include "dmachinemon/libsocket.h"
#include "dmachinemon/dmachinemon-libdatabase.h"
#include "dmachinemon/dmachinemon-commandlineparser.h"
#include "config.h"

#define SLEEPTIME (1000000 * 3)

static void
dm_sendfloat(const char * tag, float f)
{
  char * buf = NULL;
  asprintf(&buf, "%f", f);
  dm_sendinfo(buf);
  free(buf);
}

static void
dm_sendint(const char * tag, int i)
{
  char * buf = NULL;
  asprintf(&buf, "%i", i);
  dm_sendinfo(buf);
  free(buf);
}

static void
dm_sendlong(const char * tag, long i)
{
  char * buf = NULL;
  asprintf(&buf, "%li", i);
  dm_sendinfo(buf);
  free(buf);
}

static void
read_machineload(void)
{
#ifdef HAVE_GETLOADAVG
  double lav[3];
  if (getloadavg(lav, 3) == -1)
    return;

  dm_sendfloat("load1", lav[1]);
  dm_sendfloat("load2", lav[2]);
  dm_sendfloat("load3", lav[3]);
#else
  /* it probably needs the LINUX way */
  FILE*in = fopen ("/proc/loadavg", "r");
  float lav1, lav2, lav3;  
  if(!in) return ;
  fscanf(in, "%f%f%f", &lav1, &lav2, &lav3);
  dm_sendfloat("load1", lav1);
  dm_sendfloat("load2", lav2);
  dm_sendfloat("load3", lav3);
  fclose(in);
#endif
}

static void
read_machinemem(void)
{
  FILE*in = fopen ("/proc/meminfo", "r");
  int total, used, free, shared, buffers, cached;  
  if(!in) return ;
  fscanf(in, "        total:    used:    free:  shared: buffers:  cached:\nMem:%i%i%i%i%i%i", &total, &used, &free, &shared, &buffers, &cached);
  
  dm_sendint("memtotal", total);
  dm_sendint("memused", used);
  dm_sendint("memfree", free);
  dm_sendint("memshared", shared);
  dm_sendint("membuffers", buffers);
  dm_sendint("memcached", cached);
  fclose(in);
}

static void
read_topuser(void)
{
  /* this portion of code is Linux specific, using linux "ps", and fread. */
  /* FIXME: Linux specific */
  char* buf =NULL;  
  FILE*in = popen ("ps aux --no-headers | sort -k 3,3nr | head -1 | cut -b 64-", "r");
  if (in)
    {
      if (1==fscanf(in, "%as", &buf))
	{
	  dm_sendinfo("topprocess", buf);
	  free(buf);  
	}
      pclose(in); 
    }
  else
    fprintf (stderr, "dmachinemon: ps aux could not be opened\n");
  
  in = popen ("ps aux --no-headers | sort -k 3,3nr | head -1 | cut -d' ' -f1", "r");
  if (in)
    {
      if (1==fscanf(in, "%as", &buf))
	{
	  dm_sendinfo("topuser", buf);
	  free(buf);  
	}
      pclose(in); 
    }
  else
    fprintf (stderr, "dmachinemon: ps aux could not be opened\n");
}


static void
write_currenttime(void)
{
  time_t t ;

  t=time(NULL);
  dm_sendlong("time", (long)t);
}

static void 
nodeinfo_write_out (void)
{
  char * c = dm_gethostname_versatile();
  if (!c)
    fprintf(stderr, "dmachinemon: gethostname_versatile failed, expect something fatal\n");

  read_machineload();
  read_machinemem();
  read_topuser();
  write_currenttime();
  free (c);
}


void
processing_command (void)
{
  while (1)
    {
      nodeinfo_write_out();
      
      usleep ( SLEEPTIME );
    }
}

int
main(int ac, char ** av)
{ 
  int t = time(NULL);
  
  dm_Init(&ac, &av);
  processing_command();
  dm_Finalize();
  
  return 0;
}
