/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * server/client P2P node.
 */
/* 
 * 2001 Jun 9
 *  Junichi Uekawa
 * $Id: dmachinemon-servent.c,v 1.34 2002/05/22 11:11:17 dancer Exp $
 */
/* Derived from : 
 * nantarad client... obtains information from nantarad, and returns 
 * the value obtained to stdout and exit value.
 * on error, returns EXIT_FAILURE on normal failure, and -1 on 
 * more obscure failures.
 *
 * originally a Telnet clone which does little input/output cooking.
 * 2001 Feb 28 Junichi Uekawa <dancer@debian.org>
 * 2001 Mar 4 Junichi Uekawa <dancer@debian.org> reuse for nantaraclient
 * copyright 2001 Junichi Uekawa
 * See GPL version 2 or later for license.
 */
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <string.h>
#include "config.h"
#include "dmachinemon/libsocket.h"
#include "dmachinemon/dmachinemon-libnodeinfo.h"
#include "dmachinemon/dmachinemon-libdatabase.h"
#include "dmachinemon/dmachinemon-servent-libroute.h"
#include "dmachinemon/dmachinemon-commandlineparser.h"
#include "dmachinemon/dmachinemon-master.h"

/* variable declarations */
static int MAXRETRIES= 0;
static int timesleep = 0;
static int max_seen_by_depth = 0;

static dm_commandoption cdat;

static int connect_to_host_with_retry(const char * uplink_host, const char * uplink_port)
{
  int socket;
  int retries=0;

  while (-1 == (socket = dm_connect_to_host (uplink_host, atoi(uplink_port))))
    {
      if (retries ++ > MAXRETRIES)
	{
	  fprintf(stderr, "Connection failed for [%s:%s], max retry count reached. Falling back to other means \n",
		  uplink_host, uplink_port);
	  return -1;
	}
      fprintf (stderr, "Connection failed trying to connect to [%s] port [%s]. Retrying... \n", 
	       uplink_host, uplink_port);
      usleep (1000000 * 5);
    }
  return socket;
}

/* 
 * The loop to obtain the machine's data, and sending it "upwards"
 *  -- FIXME/discussion: should nodeinfo_write_out be changed to add data to client?
 */
static void
do_main_loop ( const dm_commandoption* cdat )
{
  FILE* f ;
  int socket;
  char * uplink_host = strdup(cdat->parenthostname);

  while (1)
    {
      socket = connect_to_host_with_retry (uplink_host, cdat->port_nodes);
      
      if ((socket != -1) && dm_obtain_route(socket, 1, uplink_host) && (!cdat->uplink_static))
	{
	  /* if uplink relinkage is required, and socket was open, 
	     and I want relinkage to happen, do it.
	   */
	  close (socket);
	  free ( uplink_host );
	  uplink_host = dm_obtain_uplink_host();
	  if (! uplink_host)
	    fprintf (stderr, "Error: serious problem in uplink_host handling\n");
	  if (cdat->debuglevel > 100 ) fprintf(stderr, "DEBUG: Relinkage going on: to %s \n", uplink_host);
	  continue;
	}
      if (-1 == socket)
	{
	  free ( uplink_host );
	  if ((!cdat->uplink_static) && dm_remove_last_route_to())
	    {
	      uplink_host = dm_obtain_uplink_host(); 
	      if (cdat->debuglevel > 100 ) fprintf(stderr, "DEBUG: Re-routing, removing the last route_to. Relinking to %s\n", uplink_host);
	      continue;
	    }
	  else
	    {
	      fprintf (stderr, "Relinkage options have ran out, and I could not connect to any of my uplinks.\n"
		   "Cannot continue. Dying\n");
	      exit (EXIT_FAILURE);
	    }
	  
	}

      f = (socket!=-1)?fdopen(socket, "w+"):NULL;
      if (!f)
	{
	  fprintf (stderr, "Error: Some weird error in system, cannot open the already open socket as a file stream. \n"
		   "Out of memory?\n");
	  exit (EXIT_FAILURE);
	}
      
	      
      if(cdat->debuglevel > 100)
	{
	  char * routeinfo = dm_get_route();
	  fprintf(stderr, "DEBUG: Routing information received: %s\n", routeinfo );
	  free (routeinfo);
	}

      /* before clearing the seen-buffers, use the info to show the number of machines existing. */
      if (!cdat->downlink_static)
	dm_randomly_change_route_to (&maininfo, cdat);

      /* send out the info and clear the seen-buffers */
      dm_nodeinfo_write_out(f, cdat);
      master_send_info_to_client(f, &maininfo, 1, max_seen_by_depth, cdat); /* send info to uplink. */
      fclose(f);
      usleep(1000000 * timesleep);
    }
}

/* 
 * Let the guys wait for route info.
 * Handling incoming servent connection.
 */
static int
handle_nodes(void * data)
{				/* access from nodes. */
  int t = ( (struct dm_handle_incoming_params*) data )->t;
  FILE*f; 
  int return_code;

  dm_wait_route_get();
  dm_send_route(t, &cdat);
  
  f = (t!=-1)?fdopen(t, "w+"):NULL;
  return_code = dm_process_nodes_input (f, &maininfo, NULL, NULL);
  if (f) fclose(f);
  dm_tcp_free_incoming_params(data);
  return return_code;
}

/* 
 * creating a listener process. 
 */
static void 
tcphosts(void * parameter)
{
  dm_tcp_host_setup("dmachinemon-servent", (char*)parameter,
		 (void * )handle_nodes);
}

/*
 * clients != servents. Clients are the client applications.
 */
static int 
handle_clients(void * data)
{				/* access from clients. */
  int t = ( (struct dm_handle_incoming_params*) data )->t;
  FILE*f = (t!=-1)?fdopen(t, "w"):NULL;
  int return_code;
  if (!f) 
    {
      fprintf (stderr, "Error: could not open incoming connection ? \n");
      return 0;
    }

  return_code = master_send_info_to_client(f, &maininfo, 0, max_seen_by_depth, &cdat);
  fclose(f);
  dm_tcp_free_incoming_params(data);
  return return_code;
}

/*
  Creating a server interface for clients, giving out info if accessed to this port.
*/
static void
clientserver (void * parameter)
{
  dm_tcp_host_setup("dmachinemon-servent", (char*)parameter,
		 (void*) handle_clients);
}

int 
main (int ac, char ** av)
{
  pthread_t thread ;
  time_t t = time(NULL);


  dmachinemon_parse_options(ac, av, &cdat);
  if (!cdat.parenthostname)
    {
      fprintf (stderr, 
	       "dmachinemon-servent v. %s \n\n"
	       "Not enough command-line information specified\n\n"
	       ,VERSION
	       );
      dmachinemon_print_commandline_help();
      return 1;
    }
  
  srandom (t);
  
  fprintf (stderr, "dmachinemon-servent v. %s starting\n", VERSION);

  /* initialization of variables */
  timesleep = cdat.sleeptime;
  MAXRETRIES = cdat.max_num_reconnect;
  max_seen_by_depth = cdat.number_of_hosts_of_seenby;
  
  pthread_create(&thread, NULL, (void*)tcphosts, cdat.port_nodes );
  pthread_create(&thread, NULL, (void*)clientserver, cdat.port_client);

  do_main_loop ( &cdat );  

  exit (EXIT_SUCCESS);
}
