/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Routing info library.
 *
 * policy: static functions may not do the linking. ???locking???
 */
/* 
 * This part of the code is the most hairy part of dmachinemon,
 * and also the essential part. Be careful when modifying this file.
 */
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include "dmachinemon/dmachinemon-debug.h"
#include "dmachinemon/libsocket.h"
#include "dmachinemon/dmachinemon-servent-libroute.h"
#include "dmachinemon/dmachinemon-libdatabase.h"

#define ROUTE_TO "Route-To"
#define LOCALHOST "localhost"

static char * route_to = NULL;		/* the current known route information. */
static char * my_hostname = NULL;
static char * random_redirection_target = NULL; /* the target which the redirection is done towards */

/* all these vars are locked with this single var. */
static pthread_mutex_t route_to_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t route_to_wait = PTHREAD_COND_INITIALIZER;

/* this should be used as an unit, yeah */
#define BUF_UNIT_SIZE 4096
#define OBTAINROUTEFREQ 10

static int 
detect_loop_in_route_to ( const char * myroute);

/* 
   getcar

   memory obtained here need to be freed
 */
static char *
get_lastitem (const char * ch)
{
  char * b = ch?strrchr(ch, ','):NULL;
  if (b) b++;
  return b?strdup(b):(ch?strdup(ch):NULL);
}

/* 
   getcdr

   memory obtained from here need to be freed.
 */
static char*
get_itemsexceptlast (const char * ch)
{
  char * s = ch?strdup(ch):NULL;
  char * b = s?strrchr (s, ','):NULL;
  if (b) 
    *b = 0;
  else 
    {
      free (s);
      s=NULL;
    }
  return s;
}

/**
 * obtain the route information from uplink,
 * and also return 1 if it needs a relinking.
 *
 * this function is called from the "upstream linker", 
 * accessing the uplink.
 *
 * This function cannot use cdat because gatherinfo calls it.
 *
 * @return uplink changed flag. 1 if uplink needs to be changed, 0 if not.
 */
int
dm_obtain_route (int uplink_socket, /// File descriptor for already open uplink socket connection.
		 int detect_loop_flag, /// whether to detect loops.
		 const char* current_uplink /// current uplink name.
		 )
{
  char * tmp = NULL;
  char * buf = malloc (BUF_UNIT_SIZE);
  char * uplink = NULL;
  int readlen ;
  int uplink_changed = 0;

  while (0==(readlen=read(uplink_socket, buf, BUF_UNIT_SIZE)))
    ;
  
  buf[readlen] = 0;
  sscanf(buf, ROUTE_TO ": %as\n", &tmp);  /* initial route */
  free (buf);

  pthread_mutex_lock (&route_to_lock);
  if (!my_hostname) 
    my_hostname = dm_gethostname_versatile(); 

  if (detect_loop_flag && detect_loop_in_route_to(tmp))
    {
      pthread_mutex_unlock (&route_to_lock);
      fprintf (stderr, "[rejloop %s: %s]", my_hostname, tmp); /* loop debug message */
      free (tmp);
      return 1;
    }

  /* tmp contains the route_to info -- making it into route_to */
  if (route_to && (!strcmp(route_to,tmp)))
    {				/* if route_to is the same as tmp. */
      free(tmp);
    }
  else
    {
      if (route_to) free(route_to);
      route_to = tmp;
      pthread_cond_broadcast(&route_to_wait);
    }

  //fprintf (stderr, "DEBUG %s: obtaining routing info %s\n", my_hostname, route_to);
  pthread_mutex_unlock (&route_to_lock);

  /* obtain the uplink diff, if the current_uplink is avail. */
  if (current_uplink)
    {
      uplink = dm_obtain_uplink_host();
      uplink_changed = (current_uplink && uplink) ? (strcmp(current_uplink, uplink)) : 0 ;
      if (uplink) free (uplink);
    }

  return uplink_changed;
}

/* return the second from last, in comma-delim list */
static char * 
obtain_secondlast_in_commadeliminated (const char* f)
{
  char * exceptl = get_itemsexceptlast (f);
  char * l = get_lastitem (exceptl);
  
  if (exceptl) free (exceptl);
  if (!l)
    fprintf (stderr, "DEBUG: Fatal error in logic, no secondlast\n");
  return l;
}

/**
   Get the hostname of grandparent host.

   @return The hostname of grandparent link. The memory obtained here need to be freed.
 */
char *
dm_obtain_grandparent_host(void)
{
  char * s;
  
  pthread_mutex_lock(&route_to_lock);
  s = obtain_secondlast_in_commadeliminated(route_to);
  pthread_mutex_unlock(&route_to_lock);
  return s;  
}



/**
 *   get the uplink host. (the last member in the route_to string)
 * The host to connect to for uplink.
 */
char *
dm_obtain_uplink_host (void)
{
  char * uplink = NULL;

  pthread_mutex_lock(&route_to_lock);
  if (!my_hostname) my_hostname = dm_gethostname_versatile(); /* used by master only? */
  if (route_to)
    uplink = get_lastitem(route_to);
  if (!uplink) 
    uplink = strdup(route_to);

  if ((uplink) && (!strcmp(uplink, my_hostname)))
    /* check if my uplink is being reconnected, and he wants
				      me to connect to his uplink. */
    {
      free (uplink);
      uplink = obtain_secondlast_in_commadeliminated(route_to);
      fprintf(stderr,"DEBUG: %s: my uplink sends me myself, I shall relink [route:%s -> uplink:%s].\n",
	      my_hostname,
	      route_to,
	      uplink);
    }

  pthread_mutex_unlock (&route_to_lock);
  return uplink;
}


/**
 * send the route info to downlink.
 *
 * @version 0.12, soname 4 has different interface from older version. Adds an extra parameter field, dm_commandoption *.
 */
void 
dm_send_route (int socket_up /** File descriptor for socket stream. */,
	       const dm_commandoption * cdat
	       )
{
  char * buf = NULL;

  pthread_mutex_lock(&route_to_lock);
  if (!my_hostname) 
    my_hostname = dm_gethostname_versatile(); /* should be used by master only */

  if (random_redirection_target)
    {
      /* send a different route. */
      
      asprintf(&buf, ROUTE_TO ": %s%s%s\n", route_to?route_to:"", 
	       route_to?",":"",
	       random_redirection_target
	       );
      if (cdat->debuglevel > 2 )
	fprintf (stderr, "DEBUG:[%s] redir [%s]\n", my_hostname, buf);
      free (random_redirection_target);
      random_redirection_target = NULL;
    }
  else
    {
      asprintf(&buf, ROUTE_TO ": %s%s%s\n", route_to?route_to:"", 
	       route_to?",":"",
	       my_hostname
	       );
    }

  pthread_mutex_unlock(&route_to_lock);

  write (socket_up, buf, strlen(buf));
  free (buf);
}

/**
 * Get the route information, received from upstream
 * 
 * The returned value need to be freed.
 */
char * 
dm_get_route (void)
{
  char * route_to_tmp;

  pthread_mutex_lock (&route_to_lock);
  route_to_tmp  = strdup(route_to);
  pthread_mutex_unlock (&route_to_lock);
  return route_to_tmp;
}

/**
 * wait until the initial route is obtained from the uplink.
 */
void 
dm_wait_route_get(void)
{
  pthread_mutex_lock (&route_to_lock);
  if (!route_to)		/* if null, wait for info */
    {
      pthread_cond_wait (&route_to_wait, &route_to_lock);
    }  
  pthread_mutex_unlock (&route_to_lock);
}

/** 
 * remove the last route_to member
 */
int
dm_remove_last_route_to(void)
{
  char * m;
  
  pthread_mutex_lock (&route_to_lock);
  if (route_to)
    {
      m = get_itemsexceptlast(route_to);
      free (route_to);
      route_to = m;
    }
  pthread_mutex_unlock (&route_to_lock);
  return route_to != NULL;
}

/* return 1 if it is a direct link to be considered */
static int 
is_this_link_a_direct_link(dm_machinelist * m2, const dm_commandoption * cdat)
{
  const char * m;
  int ret = 0;
  
  pthread_mutex_lock (&m2->lock);
  m = dm_get_value_text (m2, SEEN_BY);

  if (!m)
    {if (cdat->debuglevel > 2 ) fprintf (stderr, "DEBUG: No seen-by is available, fatal error in Seen-By logic!!\n");}  
  else if (!strcmp(m, LOCALHOST))
    ret=0;			/* exception handling */
  else if (!strchr(m, ','))		/* this is not a direct link, contains a , */
    {if (cdat->debuglevel > 2 ) fprintf (stderr, "DEBUG: fatal error in Seen-By logic\n");}
  else if (!strchr(strchr(m, ',') + 1, ','))
    ret=1;
  else
    ret=0;
  pthread_mutex_unlock (&m2->lock);
  
  return ret;
}
			   

/*
 * Count the number of active downlinks.
 * Does not need external locks.
 */
static int
count_active_direct_downlinks(dm_machinelist_information *m, const dm_commandoption * cdat)
{
  int count = 0;
  dm_machinelist * m2;
  
  pthread_mutex_lock(&m->machinedb_lock);
  m2=m->machinedb;
  pthread_mutex_unlock(&m->machinedb_lock);  
  while (m2)
    {
      if ((!m2->dataseenflag)	/* if the data is not seen. */
	  && (m2->clientseencount < CLIENTSEENCOUNT_THRESHOLD) 
	  && (is_this_link_a_direct_link(m2,cdat))) /* and the data is not too old. */
	count ++ ;		/* the downlink is rather active... */
      m2=m2->next;
    }
  return count;
}

/* 
 * returns 1 if route_to is looping to myself.
 * I need to have these locks locked, please!!
 */
static int
detect_loop_in_route_to ( const char * my_route_to)
{
  char * m;
  int duplicate = 0;
  
  if (!my_hostname) my_hostname = dm_gethostname_versatile(); 
  m = get_itemsexceptlast(my_route_to);
  if (m)
    {
      duplicate = detect_loop_in_route_to(m);
      free(m);
    }
  m = get_lastitem (my_route_to);
  if (!strcmp(m, my_hostname))
    duplicate ++ ;
  free (m);
  return duplicate;
}


/** 
 * Redirection target chooser.
 * Used by the uplinks to relink the downlinks.
 *
 * @version 0.6, soname 1 has different interface from older version. Adds an extra parameter field, dm_commandoption *.
 */
void
dm_randomly_change_route_to (dm_machinelist_information * m /** List of machines to choose the redirection target */,
			     const dm_commandoption * cdat)
{
  int number = count_active_direct_downlinks(m,cdat);
  int randomnum ;
  dm_machinelist *m2;
  if (number)
    randomnum = random () % number;
  else
    randomnum = 0;

  pthread_mutex_lock(&m->machinedb_lock);
  m2=m->machinedb;
  pthread_mutex_unlock(&m->machinedb_lock);

  if (number > cdat->downlinknum_threshold)
    {
      if (cdat->debuglevel > 2 ) fprintf(stderr, "DEBUG: more than %i downlinks, going\n", cdat->downlinknum_threshold);
      /* if more than four downlink exist, relink it. */
      while (randomnum)
	{
	  if ((!m2->dataseenflag)
	      && (m2->clientseencount < CLIENTSEENCOUNT_THRESHOLD)
	      && (is_this_link_a_direct_link(m2,cdat) ))
	    {
	      randomnum -- ;
	      if (!randomnum) break;
	    }
	  m2=m2->next;
	  if (!m2)
	    {
	      if (cdat->debuglevel > 2 ) fprintf (stderr, "DEBUG: fatal error in logic in randomly_change_route_to");
	      return ;
	    }
	}
      if (cdat->debuglevel > 2 ) fprintf (stderr, "DEBUG: Randomly-change-route-to is set to %s\n", m2->machinename);
      pthread_mutex_lock(&route_to_lock);
      if (random_redirection_target)
	{
	  if (cdat->debuglevel > 2 ) fprintf (stderr, "DEBUG: random-redirection-target is not NULL\n");
	  free (random_redirection_target);
	}
      random_redirection_target = strdup (m2->machinename);
      pthread_mutex_unlock(&route_to_lock);
    }
  else
    {
      if (cdat->debuglevel > 2 ) fprintf (stderr, "DEBUG: Only %i hosts available", number);
    }
  
}




