/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*
 * database updating routine.
 */
/* 
 * Master node for access. : port for nodes,  port for accessors.
 * 2001 June 9
 *  Junichi Uekawa
 * $Id: dmachinemon-libdatabase.c,v 1.27 2002/05/26 08:42:58 dancer Exp $
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include "dmachinemon/dmachinemon-debug.h"
#include "dmachinemon/libsocket.h"
#include "dmachinemon/dmachinemon-libdatabase.h"
//#include "dmachinemon/dmachinemon-master.h"


/**
   The generic information database, which is used for 
   default operations.
 */
dm_machinelist_information maininfo = {NULL, PTHREAD_MUTEX_INITIALIZER};
				/* this is the generic database. */


/**
 * update a DB of associated-string 
 * used to update a database for one machine.
 */
void
dm_update_db (dm_machinelist* mine, dm_pairstring* ps,
		const char * category, const char * string)
{
  if (!ps)
    {				/* There is no matching pairstring data available, malllc! */
				/* this ps is local data... */
      ps = malloc(sizeof(dm_pairstring));
      ps -> stringtitle = strdup(category);
      ps -> stringdata = strdup (string);
      pthread_mutex_init (&(ps->lock), NULL);  
				/* ps is global data now... */
      pthread_mutex_lock(&(mine->lock));
      ps -> next = mine -> data;
      mine -> data = ps;
      pthread_mutex_unlock(&(mine->lock));
      //fprintf (stderr, "DEBUG: new data\n");
    }
  else if (!strcmp (category, ps->stringtitle))
    {				/* match, replace. */
      pthread_mutex_lock(&(ps->lock));
      free (ps->stringdata);
      ps->stringdata=strdup(string);
      pthread_mutex_unlock(&(ps->lock));
    }
  else
    {				/* try next. */
      dm_update_db (mine, ps->next, category, string);
    }
}

/**
   Get a text value from the database for one machine.

   @return the string, or NULL on failure.
   @version 0.12 changed the interface.
 */
const char * 
dm_get_value_text(dm_machinelist * now,/// database
		  const char * title /// key
		  )
{
  dm_pairstring * p = now->data;
  while(p)
    {
      if(strcmp(p->stringtitle, title))
	p=p->next;
      else
	break;
    }

  if(p)
    return p->stringdata;
  
  return 0;
}

/**
   Get value from database in floating point format.

   @return floating point value, or 0 on failure.
 */
float dm_get_value_float(dm_machinelist*now,///database
			 const char*title /// key
			 )
{
  const char * buf=dm_get_value_text(now, title);  
  if (!buf)
    return 0;
  else
    return atof(buf);  
}


/** searches for a machine in the DB 
    @return NULL if the machine is not already in DB */
dm_machinelist * 
dm_search_machine (dm_machinelist_information * machinedbinfo, // machine information database
		   const char * name // Machine hostname to look up.
		   )
{
  dm_machinelist * new = machinedbinfo->machinedb;
  for (new = machinedbinfo->machinedb; new; new = new -> next)
    {
      if (!strcmp (new -> machinename, name))
	break;
    }
  return new ;  
}

/**
   Count the number of nodes and return.

   @return number of nodes in the DB
 */
int 
dm_count_nodes_number (dm_machinelist_information * machinedbinfo)
{
  dm_machinelist * my = machinedbinfo -> machinedb;
  int count = 0;
  
  while (my)
    {
      count ++ ;
      my = my -> next;
    }
  return count;
}


/**
   Process the input from node, to refresh my data in database.
   The createapplicationdatacallback can be NULL, if there is no
   need for a callback function.
   The callback function will be called with the data_for_callback, and the hostname information.

   This function is very hairy.

   This function cannot use cdat information, because gatherinfo calls this function.
   
   @return 0 on success, 1 on error.
*/
int 
dm_process_nodes_input (FILE*f, /// File pointer to input stream.
			dm_machinelist_information * machinedbinfo, /// machine db information 
			void* (*createapplicationdatacallback)(void*, const char* ),
			void * data_for_callback /// the data sent to the callback function
			)
{
  dm_machinelist * nowdb = NULL; /* the current data to operate upon. */
  char * category = NULL;
  char * stringdata = NULL;  
  char * buf = NULL;
  int buflen = 0;

  while(-1!=getline(&buf, &buflen, f))
    {
      if (*buf == ':')
	{			/* start of a new machine entry, which might or might not already 
				   exist in the database */
	  dm_machinelist * new = machinedbinfo->machinedb;

	  /* obtain the name for system, as stringdata  :HOSTNAME: */
	  stringdata = strdup (buf + 1);
	  {
	    char * pos = strchr (stringdata, ':');
	    if (pos)
	      *pos = 0;		/* remove the final ":" */
	    else
	      {			/* error handling */
		fprintf (stderr, "dmachinemon-libdatabase.c: invalid information packet received [%s]\n", buf);
		free (stringdata);
		if (buf) free(buf);
		return 1;
	      }
	  }
	  
	  new = dm_search_machine (machinedbinfo, stringdata);
	  
	  if (!new)
	    {			/* this is a new machine, 
				   create a new db entry */
	      new=malloc(sizeof(dm_machinelist));
	      if (!(new=malloc(sizeof(dm_machinelist))))
		{
		  fprintf(stderr, "out of memory error\n");
		  if (buf) free(buf);
		  return 1;
		}
	      new->data = NULL;
	      if (createapplicationdatacallback) /* I will call the callback when it is defined. */
		new->applicationdata = createapplicationdatacallback(data_for_callback, stringdata);
	      
	      pthread_mutex_init(&new->lock, NULL);
	      new->machinename = strdup(stringdata);
	      pthread_mutex_lock(& (machinedbinfo -> machinedb_lock));
	      new->next = machinedbinfo->machinedb;
	      machinedbinfo->machinedb=new;
	      pthread_mutex_unlock(& (machinedbinfo -> machinedb_lock));
	    }
	  free(stringdata);
	  nowdb = new ;		/* This is the new one to be operated to. */
	  nowdb->dataseenflag = 0;
	  nowdb->clientseencount = 0; /* this is new data */
	}
      else if (2==sscanf(buf, "%a[^:]: %as\n", &category, &stringdata))
	{			/* FIXME: LINUX specific */
	  /* 
	     Start of a non-machine entry.
	  */
	  dm_pairstring * newdata;
	  if (!nowdb)
	    {
	      fprintf (stderr, "syntax error in input, no machine spec info has been received\n");
	      if (category) free (category);
	      if (stringdata) free (stringdata);
	      free (buf);
	      return 1;	      
	    }
	  
	  if (!strcmp(category, SEEN_BY))
	    {			/* add myself to the list of machines that saw this data.
				   This is in stringdata.
				 */
	      char * tmp = NULL;
	      char * hostname = dm_gethostname_versatile();

	      asprintf(&tmp, "%s,%s", stringdata, hostname);

	      free (stringdata);
	      free (hostname);	
	      stringdata = tmp;
	    }
	  
	  pthread_mutex_lock(&nowdb->lock);
	  newdata = nowdb->data;
	  nowdb->dataseenflag = 0;
	  nowdb->clientseencount = 0; /* this is new data */
	  pthread_mutex_unlock(&nowdb->lock);
	  
	  dm_update_db(nowdb, newdata, category, stringdata);
	  free(category);
	  free(stringdata);
	}	  
      else
	{
	  fprintf (stderr, "dmachinemon: Input syntax error in parsing input from nodes. [%s]\n", buf);
	  if (category) free(category);
	  if (stringdata) free (stringdata);
	  free (buf);
	  return 1;	  
	}
      //fprintf (stderr, "DEBUG: continue\n");
    }
  if (buf) free (buf);
  buf = NULL;  
  return 0;  
}
